/*
 * Fabric Management Server
 *
 * startup is:
 *  initialize local stuff
 *  read databases
 *  wait for connections from FMAs
 *  first FMA gets asked to "buzz network"
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_fabric.h"
#include "lf_channel.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_io.h"
#include "fms_db.h"
#include "fms_fma.h"
#include "fms_notify.h"

struct fms_vars F;

/*
 * local prototypes
 */
static int fms_init_vars(void);
static void fms_parse_options(int argc, char **argv);
static void init_signals(void);
static void fms_init_logfile(void);
static void fms_load_environment(void);

static void
usage()
{
  fprintf(stderr, "Usage: fms\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -D - increment debug level\n");
  fprintf(stderr, "  -R <fms_run> - specify FMS_RUN directory\n");
  fprintf(stderr, "  -N <fms_db_name> - specify database name\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(int argc,
     char **argv)
{
  uint32_t ms;
  int rc;

  /* initialize libfma */
  lf_init();

  /* Initialize the error handling subsystem before we do anything else */
  fms_init_error_vars();
  fms_init_error();

  /* initialize error handler */
  lf_set_error_handler(fms_save_error);

  /* initialize all variables to default values */
  if (fms_init_vars() != 0) LF_ERROR(("initializing variables"));

  /* first load environmental settings, then command line may override */
  fms_load_environment();
  fms_parse_options(argc, argv);
  fms_init_logfile();

  if (init_alerts() != 0) LF_ERROR(("initializing alerts"));
  if (init_database() != 0) LF_ERROR(("initializing database"));
  fms_init_settings();
  if (init_io() != 0) LF_ERROR(("initializing I/O"));
  if (init_fma() != 0) LF_ERROR(("initializing FMA subsystem"));
  if (init_fabric() != 0) LF_ERROR(("Initializing fabric system"));

  init_signals();	/* signal handlers do things by scheduling events */

  /*
   * Get an initial idea of the fabric
   */
  if (load_fabric() != 0) LF_ERROR(("loading fabric"));

  /* start monitoring all the switches */
  if (init_switch_monitoring() != 0) {
    LF_ERROR(("Initializing switch monitoring system"));
  }

  fms_notify(FMS_EVENT_INFO, "%s FMS up and running", Lf_version);

  /*
   * Just loop handling events forever
   */
  while (!F.done) {
    ms = lf_check_events();	/* look for any scheduled events */

    /* don't sleep forever, since threads may request tasks */
    if (ms < 0 || ms > FMS_MIN_POLL_WAIT) {
      ms = FMS_MIN_POLL_WAIT;
    }

    /* look for any pending I/O requests */
    rc = lf_poll_channels(ms);
    if (rc == -1) LF_ERROR(("Polling for I/O"));
  }

  close_io();
  close_database();

  fms_exit(0);

 except:
  fms_perror();
  fms_exit(1);
  return 0;
}

/*
 * Initialize the FMS global data structure
 */
static int
fms_init_vars()
{
  /* randomize random number generator */
  srandom((int)time(NULL));

  /* initialize FMA variables */
  F.done = FALSE;

  F.fms_run = LF_DFLT_FMS_RUN;
  F.db_name = LF_DFLT_DB_NAME;

  /* vars associated with notification requests */
  if (init_notify_vars() != 0) LF_ERROR(("Initializing notify vars"));

  /* vars associated with alert subsystem */
  if (init_alert_vars() != 0) LF_ERROR(("Initializing alert vars"));

  /* vars for the I/O subsystem */
  if (init_io_vars() != 0) return -1;

  /* vars for the fabric subsystem */
  if (init_fabric_vars() != 0) return -1;

  /* vars for the switch monitoring subsystem */
  if (init_switch_vars() != 0) return -1;

  /* vars for the FMA subsystem */
  if (init_fma_vars() != 0) return -1;

  /* variables for tracking and controlling FMS state */
  fms_init_state_vars();

  /* user-modifiable persistant settings */
  fms_init_settings_vars();

  return 0;

 except:
  return -1;
}

/*
 * load environment variables
 */
static void
fms_load_environment()
{
  char *tp;

  /* override database directory? */
  tp = getenv(LF_ENV_FMS_RUN);
  if (tp != NULL && *tp != '\0') {
    F.fms_run = tp;
  }

  /* override database name? */
  tp = getenv(LF_ENV_DB_NAME);
  if (tp != NULL && *tp != '\0') {
    F.db_name = tp;
  }

  /* enable debug? */
  tp = getenv(LF_ENV_FMS_DEBUG);
  if (tp != NULL && *tp != '\0') {
    F.debug = TRUE;
  }
}

/*
 * parse command line args
 */
static void
fms_parse_options(
  int argc,
  char **argv)
{
  extern char *optarg;
  int c;

  while ((c = getopt(argc, argv, "hDH:R:N:V")) != EOF) switch (c) {
  case 'h':
    usage();
    break;
  case 'H':
    fprintf(stderr,
	"Please note: -H has been replaced with -R and is deprecated\n");
    /* FALLSTHROUGH */
  case 'R':
    F.fms_run = optarg;
    break;
  case 'N':
    F.db_name = optarg;
    break;
  case 'D':
    ++F.debug;
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }
}

/*
 * initialize signals
 * ignore SIGPIPE since we are messing with sockets
 */
void
init_signals()
{
  struct sigaction sa;
  int rc;

  /* Ignore SIGPIPE */
  memset(&sa, 0, sizeof(sa));
  sa.sa_handler = SIG_IGN;
  rc = sigaction(SIGPIPE, &sa, NULL);
  if (rc != 0) {
    perror("Error ignoring SIGPIPE");
    exit(1);
  }
}

/* print and exit */
void
fms_perror_exit(
  int code)
{
  fms_perror();
  if (F.debug) {
    abort();
  } else {
    fms_exit(code);
  }
}

/*
 * perform any cleanup and exit
 */
void
fms_exit(
  int code)
{
  exit(code);
}


/*
 * set up logging of events to a file
 */
static void
fms_init_logfile()
{
  FILE *fp;
  lf_string_t log;
  int id;

  /* set up for events to stderr */
  id = register_notify_file(
      (FMS_EVENT_ALL & ~FMS_EVENT_DEBUG)
	| (F.debug ? FMS_EVENT_DEBUG : 0),
      stderr);
  if (id == -1) {
    fprintf(stderr, "Cannot set up stderr error client\n");
    exit(1);

  }

  sprintf(log, "%s/fms.log", F.fms_run);
  fp = fopen(log, "w");
  if (fp == NULL) LF_ERROR(("Cannot open logfile \"%s\"", log));
  id = register_notify_file((FMS_EVENT_ALL & ~FMS_EVENT_DEBUG)
			    | (F.debug ? FMS_EVENT_DEBUG : 0),
			    fp);
  if (id == -1) LF_ERROR(("Error registering log notification"));

  return;

 except:
  fms_perror();
}
